/*
   This program is provided under the LGPL license ver 2.1.
   Written by Katsumi.
   http://hp.vector.co.jp/authors/VA016157/
   kmorimatsu@users.sourceforge.jp
*/

#include "LPC8xx.h"

#include "./prototypes.h"
#include "./config.h"

// Prototype
const unsigned char ps2table[];

static int g_ps2clk=1;
static int g_ps2dat=0;

inline void allow_ps2(void){
	// Allow the signal from PS/2 keyboard
	LPC_GPIO_PORT->DIR0 &= ~(1<<5); // PIO0_5: input
}
inline void disallow_ps2(void){
	// Disallow the signal from PS/2 keyboard
	LPC_GPIO_PORT->DIR0 |= (1<<5); // PIO0_5: output
	LPC_GPIO_PORT->CLR0 = (1<<5);  // Output L
	g_ps2dat=0;
}
inline void check_ps2(void){
	// This function must be the most simple.
	int ps2clk = LPC_GPIO_PORT->PIN0 & (1<<5); // Check P0.5
	if (g_ps2clk) {    // When the signal just before reading P0.5 is H
		if (!ps2clk) { // and the signal of P0.5 is L.
			// Shift-right the data
			g_ps2dat=g_ps2dat>>1;
			// Data comes from P0.1.
			// Note that data is inversed to have first bit of H.
			g_ps2dat|=(LPC_GPIO_PORT->PIN0 & (1<<1)) ? 0:(1<<10);
		}
	}
	g_ps2clk=ps2clk;
}
inline void completed_ps2(void){
	static int counter=0;
	static int breakcode=0;
	static int shiftkey=0;
	unsigned char data,ascii;
	if (g_ps2dat & 1) {
		// 11 bit data are all received.
		// Note that data is inversed (see check_ps2() function).
		data=((~g_ps2dat)>>1)&0xff;
		g_ps2dat=0;
		ascii=0;
		switch(data){
		case 0xf0: // Break code
			breakcode=1;
			break;
		case 0xe0: // Extension (ignore it here)
			break;
		case 0x12: case 0x59: // Shift key
			if (breakcode) shiftkey=0;
			else shiftkey=0x80;
			breakcode=0;
			break;
		default:
			if (0x7f<data) break;
			ascii=ps2table[shiftkey+data];
			break;
		}
		if (ascii) {
			if (breakcode) {
				breakcode=0;
			} else {
				// Valid ascii code was received.
				//print_char(ascii);
				UARTSend(ascii);
			}
		}
	}
	if (g_ps2dat) {
		// When receiving data, run the counter.
		counter++;
		if (counter & (1<<6)) {
			// Data is invalid when counter reaches certain value.
			// Counter must be less than 8 according to PS/2 and NTSC signal speeds.
			counter=0;
			g_ps2dat=0;
		}
	} else {
		// When there is no data, reset counter.
		counter=0;
	}
}


/*
 * PS/2 -> ASCII data conversion table.
 * 101 and 106 keyboards are supported.
 */
const unsigned char ps2table[]={
#ifdef KEYBOARD101
		// 101
		// Shift key up
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x60,0x00,
		0x00,0x00,0x00,0x00,0x00,0x71,0x31,0x00,0x00,0x00,0x7a,0x73,0x61,0x77,0x32,0x00,
		0x00,0x63,0x78,0x64,0x65,0x34,0x33,0x00,0x00,0x20,0x76,0x66,0x74,0x72,0x35,0x00,
		0x00,0x6e,0x62,0x68,0x67,0x79,0x36,0x00,0x00,0x00,0x6d,0x6a,0x75,0x37,0x38,0x00,
		0x00,0x2c,0x6b,0x69,0x6f,0x30,0x39,0x00,0x00,0x2e,0x2f,0x6c,0x3b,0x70,0x2d,0x00,
		0x00,0x00,0x27,0x00,0x5b,0x3d,0x00,0x00,0x00,0x00,0x0a,0x5d,0x00,0x5c,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x11,0x00,0x00,0x00,0x00,
		0x00,0x7f,0x14,0x00,0x13,0x12,0x1b,0x00,0x00,0x2b,0x00,0x2d,0x2a,0x00,0x00,0x00,
		// Shift key down
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x7e,0x00,
		0x00,0x00,0x00,0x00,0x00,0x51,0x21,0x00,0x00,0x00,0x5a,0x53,0x41,0x57,0x40,0x00,
		0x00,0x43,0x58,0x44,0x45,0x24,0x23,0x00,0x00,0x20,0x56,0x46,0x54,0x52,0x25,0x00,
		0x00,0x4e,0x42,0x48,0x47,0x59,0x5e,0x00,0x00,0x00,0x4d,0x4a,0x55,0x26,0x2a,0x00,
		0x00,0x3c,0x4b,0x49,0x4f,0x29,0x28,0x00,0x00,0x3e,0x3f,0x4c,0x3a,0x50,0x5f,0x00,
		0x00,0x00,0x22,0x00,0x7b,0x2b,0x00,0x00,0x00,0x00,0x0a,0x7d,0x00,0x7c,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x00,0x11,0x00,0x00,0x00,0x00,
		0x00,0x7f,0x14,0x00,0x13,0x12,0x1b,0x00,0x00,0x2b,0x00,0x2d,0x2a,0x00,0x00,0x00,
#endif
#ifdef KEYBOARD106
		// 106
		// Shift key up
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x60,0x00,
		0x00,0x00,0x00,0x00,0x00,0x71,0x31,0x00,0x00,0x00,0x7a,0x73,0x61,0x77,0x32,0x00,
		0x00,0x63,0x78,0x64,0x65,0x34,0x33,0x00,0x00,0x20,0x76,0x66,0x74,0x72,0x35,0x00,
		0x00,0x6e,0x62,0x68,0x67,0x79,0x36,0x00,0x00,0x00,0x6d,0x6a,0x75,0x37,0x38,0x00,
		0x00,0x2c,0x6b,0x69,0x6f,0x30,0x39,0x00,0x00,0x2e,0x2f,0x6c,0x3b,0x70,0x2d,0x00,
		0x00,0x5c,0x3a,0x00,0x40,0x5e,0x00,0x00,0x00,0x00,0x0a,0x5b,0x00,0x5d,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x5c,0x11,0x00,0x00,0x00,0x00,
		0x00,0x7f,0x14,0x00,0x13,0x12,0x1b,0x00,0x00,0x2b,0x00,0x2d,0x2a,0x00,0x00,0x00,
		// Shift key down
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x09,0x7e,0x00,
		0x00,0x00,0x00,0x00,0x00,0x51,0x21,0x00,0x00,0x00,0x5a,0x53,0x41,0x57,0x22,0x00,
		0x00,0x43,0x58,0x44,0x45,0x24,0x23,0x00,0x00,0x20,0x56,0x46,0x54,0x52,0x25,0x00,
		0x00,0x4e,0x42,0x48,0x47,0x59,0x26,0x00,0x00,0x00,0x4d,0x4a,0x55,0x27,0x28,0x00,
		0x00,0x3c,0x4b,0x49,0x4f,0x00,0x29,0x00,0x00,0x3e,0x3f,0x4c,0x2b,0x50,0x3d,0x00,
		0x00,0x5f,0x2a,0x00,0x60,0x7e,0x00,0x00,0x00,0x00,0x0a,0x7b,0x00,0x7d,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x08,0x00,0x00,0x00,0x7c,0x11,0x00,0x00,0x00,0x00,
		0x00,0x7f,0x14,0x00,0x13,0x12,0x1b,0x00,0x00,0x2b,0x00,0x2d,0x2a,0x00,0x00,0x00,
#endif
};

